/*:
 * @target MZ
 * @plugindesc v1.8 コモンイベントのメッセージ抽出→TXT出力（メッセージごと1行空け／本文の \N[n] を正しく展開） 
 * @author HS
 *
 * @param outputFolder
 * @text 出力フォルダ
 * @default exports
 *
 * @param includeChoices
 * @text 選択肢を含める
 * @type boolean
 * @default true
 *
 * @param includeScrollText
 * @text スクロール文を含める
 * @type boolean
 * @default true
 *
 * @param replaceVariables
 * @text \V[n]を置換
 * @type boolean
 * @default false
 *
 * @param enableShorthandN
 * @text 簡易N表記を許可（\N1 / N1）
 * @type boolean
 * @default false
 *
 * @param includeComments
 * @text コメント行を含める
 * @type boolean
 * @default false
 *
 * @param appendIdIfDuplicate
 * @text 同名時にID付与
 * @type boolean
 * @default true
 *
 * @param appendTimestamp
 * @text タイムスタンプ付与
 * @type boolean
 * @default false
 *
 * @param extraBlankBetweenLines
 * @text 行間を1行空ける
 * @type boolean
 * @default true
 * @desc 「文章の表示」ブロックの区切りごとに1行の空白を入れます（台本読みやすさ向上）。
 *
 * @param speakerDelimiter
 * @text 話者区切り記号
 * @default ：
 *
 * @command exportAll
 * @text すべて出力
 *
 * @command exportRange
 * @text 範囲出力
 * @arg fromId @text 開始ID @type number @min 1 @default 1
 * @arg toId   @text 終了ID @type number @min 1 @default 9999
 *
 * @command exportOne
 * @text ID指定で1件出力
 * @arg id @text コモンイベントID @type number @min 1 @default 1
 *
 * @command pickAndExport
 * @text 1件選んで出力（ピッカー）
 */

(() => {
  "use strict";
  const PLUGIN_NAME = "HS_ExportCommonText_MZ";
  const p = PluginManager.parameters(PLUGIN_NAME);

  const outputFolder           = String(p.outputFolder || "exports");
  const includeChoices         = p.includeChoices === "true";
  const includeScrollText      = p.includeScrollText === "true";
  const replaceVariables       = p.replaceVariables === "true";
  const enableShorthandN       = p.enableShorthandN === "true";
  const includeComments        = p.includeComments === "true";
  const appendIdIfDuplicate    = p.appendIdIfDuplicate === "true";
  const appendTimestamp        = p.appendTimestamp === "true";
  const extraBlankBetweenLines = p.extraBlankBetweenLines === "true";
  const speakerDelimiter       = String(p.speakerDelimiter || "：");

  // コマンド
  PluginManager.registerCommand(PLUGIN_NAME, "exportAll", () => exportCommonEvents({ mode:"all" }));
  PluginManager.registerCommand(PLUGIN_NAME, "exportRange", args =>
    exportCommonEvents({ mode:"range", fromId:Number(args.fromId||1), toId:Number(args.toId||9999) })
  );
  PluginManager.registerCommand(PLUGIN_NAME, "exportOne", args =>
    exportCommonEvents({ mode:"one", id:Number(args.id||1) })
  );
  PluginManager.registerCommand(PLUGIN_NAME, "pickAndExport", () => {
    if (!Utils.isNwjs()) return alert("ピッカーはPCテストプレイ（NW.js）で使用してください。");
    SceneManager.push(Scene_CommonEventPicker);
  });

  // 出力メイン
  function exportCommonEvents(opt){
    if (!Utils.isNwjs()) { alert("テキスト出力はPCテストプレイ（NW.js）専用です。"); return; }
    const { fs, path, outDir } = ensureOutDir(); if (!fs) return;

    const ids = collectIds(opt);
    if (!ids.length) return showToast("対象コモンイベントがありません");

    let count = 0;
    ids.forEach(id=>{
      const ce = $dataCommonEvents[id];
      if (!ce || !ce.list) return;

      const lines = [];
      const header = `# CommonEvent ${id}: ${ce.name || "(no name)"}  (${new Date().toLocaleString()})`;
      lines.push(header, "-".repeat(Math.min(header.length,120)));

      const body = extractFromCommandList(ce.list);
      if (body.length) lines.push(...body); else lines.push("(メッセージなし)");

      const fileName = buildFileName(ce.name,id) + ".txt";
      try {
        fs.writeFileSync(path.join(outDir, fileName), lines.join("\n"), "utf-8");
        count++;
      } catch(e) {
        alert("ファイル書き込みに失敗: " + (e?.message||e));
      }
    });

    showToast(`${count}件 出力しました → ${outputFolder}`);
  }

  function collectIds(opt){
    const ids=[];
    if(opt.mode==="all"){
      for(let i=1;i<$dataCommonEvents.length;i++){ if($dataCommonEvents[i]) ids.push(i); }
    }else if(opt.mode==="range"){
      const a=Math.max(1,Number(opt.fromId||1)), b=Math.max(a,Number(opt.toId||a));
      for(let i=a;i<=b;i++){ if($dataCommonEvents[i]) ids.push(i); }
    }else if(opt.mode==="one"){
      const id=Number(opt.id||0); if(id>0 && $dataCommonEvents[id]) ids.push(id);
    }
    return ids;
  }

  // 出力先（プロジェクト直下）
  function projectRootPath(){
    let p = decodeURIComponent(window.location.pathname||"");
    if (/^\/[A-Za-z]:/.test(p)) p = p.slice(1); // /C:/... → C:/...
    if (p.toLowerCase().endsWith("/index.html")) p = p.slice(0, -"/index.html".length);
    return p.replace(/[\\\/]+$/,"");
  }
  function ensureOutDir(){
    if(!Utils.isNwjs()){ alert("テキスト出力はPCテストプレイ（NW.js）専用です。"); return {fs:null,path:null,outDir:null}; }
    const fs=require("fs"), path=require("path");
    const outDir = path.join(projectRootPath(), outputFolder);
    try{ if(!fs.existsSync(outDir)) fs.mkdirSync(outDir,{recursive:true}); }
    catch(e){ alert("出力フォルダの作成に失敗: "+(e?.message||e)); return {fs:null,path:null,outDir:null}; }
    console.log("[HS_ExportCommonText_MZ] outDir:", outDir);
    return {fs,path,outDir};
  }

  //========================
  // メッセージ抽出
  //========================
  function extractFromCommandList(list){
    const out=[];
    for(let i=0;i<list.length;i++){
      const cmd=list[i]; if(!cmd) continue;

      // 文章の表示
      if(cmd.code===101){
        const nameFromField = readSpeakerFromCommand101(cmd);
        const rawLines=[]; let j=i+1;
        while(j<list.length && list[j] && list[j].code===401){
          rawLines.push(String(list[j].parameters[0]||"")); j++;
        }
        if(rawLines.length){
          if (nameFromField) {
            // ★名前欄があるときは本文の \N[n]/\P[n] を削除しない（内容として展開して残す）
            rawLines.forEach(line=>{
              const onlyStripExplicitNameTag = stripExplicitInlineNameTag(line); // \name<雪乃> だけ除去
              const body = convertEscapeForExport(onlyStripExplicitNameTag);
              out.push(body.trim().length ? `${nameFromField}${speakerDelimiter}${body}` : body);
            });
          } else {
            // ★名前欄が空のときのみ、先頭の \name<...> / \N[n] / \P[n] を“話者タグ”として扱う
            const {speaker,lines}=parseBlockSpeakerAndStrip(rawLines);
            if(speaker){ lines.forEach(line=> out.push(line.trim().length?`${speaker}${speakerDelimiter}${line}`:line)); }
            else{ lines.forEach(line=> out.push(line)); }
          }
          // ブロック終端で空行
          if(extraBlankBetweenLines && out.length && out[out.length-1] !== "") out.push("");
        }
        i=j-1; continue;
      }

      // 選択肢
      if(includeChoices && cmd.code===102){
        const choices=cmd.parameters[0]||[];
        choices.forEach((c,k)=> out.push(`【選択肢${k+1}】${convertEscapeForExport(String(c))}`));
        if(extraBlankBetweenLines && out.length && out[out.length-1] !== "") out.push("");
        continue;
      }

      // スクロールする文章
      if(includeScrollText && cmd.code===105){
        const s=String(cmd.parameters[0]||"");
        out.push(`【スクロール】${convertEscapeForExport(s)}`);
        if(extraBlankBetweenLines && out.length && out[out.length-1] !== "") out.push("");
        continue;
      }

      // コメント
      if(includeComments && cmd.code===108){
        out.push(`// ${String(cmd.parameters[0]||"")}`);
        let j=i+1;
        while(j<list.length && list[j] && list[j].code===408){
          out.push(`// ${String(list[j].parameters[0]||"")}`); j++;
        }
        if(extraBlankBetweenLines && out.length && out[out.length-1] !== "") out.push("");
        i=j-1; continue;
      }
    }
    return out;
  }

  // 名前欄から話者取得（\N[n], \P[n] も展開）
  function readSpeakerFromCommand101(cmd){
    const name = String(cmd.parameters[4]||""); // MZの名前欄
    if(!name) return "";
    let t = name.replace(/\\\\/g,"\\");
    t = t.replace(/\\N\[(\d+)\]/gi,(_,n)=>actorNameSafe(Number(n)));
    t = t.replace(/\x1bN\[(\d+)\]/gi,(_,n)=>actorNameSafe(Number(n)));
    t = t.replace(/\\P\[(\d+)\]/gi,(_,n)=>partyMemberNameSafe(Number(n)));
    t = t.replace(/\x1bP\[(\d+)\]/gi,(_,n)=>partyMemberNameSafe(Number(n)));
    return t;
  }

  // （名前欄がある時だけ使う）\name<...> 形式だけを除去。 \N[n]/\P[n] は残す
  function stripExplicitInlineNameTag(s){
    return String(s).replace(/^\\(?:n|N|name|speaker)<[^>]+>/i,"");
  }

  // （名前欄が空の時だけ使う）先頭の \name<...> / \N[n] / \P[n] を話者とみなして除去
  function parseBlockSpeakerAndStrip(blockLines){
    if(!blockLines.length) return {speaker:"",lines:[]};
    let first = blockLines[0], m;

    m = first.match(/^\\(?:n|N|name|speaker)<([^>]+)>/i);
    if(m){
      const speaker = convertEscapeForExport(m[1]);
      first = first.slice(m[0].length);
      const rest=[first,...blockLines.slice(1)].map(s=>convertEscapeForExport(s));
      return {speaker,lines:rest};
    }

    m = first.match(/^\\N\[(\d+)\]/i);
    if(m){
      const speaker = actorNameSafe(Number(m[1]));
      first = first.slice(m[0].length);
      const rest=[first,...blockLines.slice(1)].map(s=>convertEscapeForExport(s));
      return {speaker,lines:rest};
    }

    m = first.match(/^\\P\[(\d+)\]/i);
    if(m){
      const speaker = partyMemberNameSafe(Number(m[1]));
      first = first.slice(m[0].length);
      const rest=[first,...blockLines.slice(1)].map(s=>convertEscapeForExport(s));
      return {speaker,lines:rest};
    }

    if(enableShorthandN){
      m = first.match(/^\\N(\d+)/);
      if(m){
        const speaker = actorNameSafe(Number(m[1]));
        first = first.slice(m[0].length);
        const rest=[first,...blockLines.slice(1)].map(s=>convertEscapeForExport(s));
        return {speaker,lines:rest};
      }
      m = first.match(/^N(\d+)(?![A-Za-z0-9_])/);
      if(m){
        const speaker = actorNameSafe(Number(m[1]));
        first = first.slice(m[0].length);
        const rest=[first,...blockLines.slice(1)].map(s=>convertEscapeForExport(s));
        return {speaker,lines:rest};
      }
    }

    const rest = blockLines.map(s=>convertEscapeForExport(s));
    return {speaker:"",lines:rest};
  }

  // 置換（\N[n]/\P[n]/\V[n] を本文でも展開）
  function convertEscapeForExport(text){
    if(text==null) return "";
    let t = String(text).replace(/\\\\/g,"\\");
    if(replaceVariables && typeof $gameVariables!=="undefined"){
      t = t.replace(/\\V\[(\d+)\]/gi,(_,n)=> String($gameVariables.value(Number(n)) ?? 0));
      t = t.replace(/\x1bV\[(\d+)\]/gi,(_,n)=> String($gameVariables.value(Number(n)) ?? 0));
    }
    t = t.replace(/\\N\[(\d+)\]/gi,(_,n)=>actorNameSafe(Number(n)));
    t = t.replace(/\x1bN\[(\d+)\]/gi,(_,n)=>actorNameSafe(Number(n)));
    t = t.replace(/\\P\[(\d+)\]/gi,(_,n)=>partyMemberNameSafe(Number(n)));
    t = t.replace(/\x1bP\[(\d+)\]/gi,(_,n)=>partyMemberNameSafe(Number(n)));
    t = t.replace(/\r\n/g,"\n").replace(/\r/g,"\n");
    return t;
  }

  function actorNameSafe(id){
    try{
      if(window.$gameActors){
        const a = $gameActors.actor(id);
        if(a && typeof a.name==="function"){
          const nm = a.name();
          if(nm && nm.length) return nm;
        }
      }
    }catch(_){}
    const db = window.$dataActors?.[id]?.name;
    return db || `(Actor${id})`;
  }
  function partyMemberNameSafe(n){
    const idx = Number(n)-1;
    try{
      const mem = $gameParty?.members?.()[idx];
      if(mem && typeof mem.name==="function") return mem.name();
    }catch(_){}
    return `(Member${n})`;
  }

  // ファイル名など
  function buildFileName(name,id){
    let base=(name||"").trim(); if(!base) base=`CommonEvent_${id}`;
    base=base.replace(/[\\\/:\*\?"<>\|]/g,"＿").replace(/[\u0000-\u001f]/g," ").replace(/\s+/g," ").trim();
    if(base.length>200) base=base.slice(0,200);
    const {fs,path,outDir}=ensureOutDir(); let candidate=base;
    if(appendIdIfDuplicate && fs){
      let suffix=`_ID${id}`; if(appendTimestamp) suffix+=`_${timeStamp()}`;
      const full=path.join(outDir,candidate+".txt"); if(fs.existsSync(full)) candidate=base+suffix;
    }else if(appendTimestamp){ candidate=`${base}_${timeStamp()}`; }
    return candidate;
  }
  function timeStamp(){ const d=new Date(),z=n=>String(n).padStart(2,"0");
    return `${d.getFullYear()}${z(d.getMonth()+1)}${z(d.getDate())}_${z(d.getHours())}${z(d.getMinutes())}${z(d.getSeconds())}`;
  }

  function showToast(text){
    const scene=SceneManager._scene; if(!scene) return;
    const spr=new Sprite(new Bitmap(Graphics.width,48));
    spr.bitmap.fontSize=18; spr.bitmap.drawText(String(text),0,0,Graphics.width,48,"center");
    spr.x=0; spr.y=0; spr.opacity=0; scene.addChild(spr);
    let t=0; spr.update=()=>{ t++; if(t<12) spr.opacity=Math.min(255,spr.opacity+25);
      else if(t<120) spr.opacity=255; else if(t<140) spr.opacity=Math.max(0,spr.opacity-25); else scene.removeChild(spr); };
  }

  // ピッカー
  class Scene_CommonEventPicker extends Scene_MenuBase{
    create(){ super.create(); this.createHelpWindow();
      this._listWindow=new Window_CommonEventList(0,this._helpWindow.height,Graphics.boxWidth,Graphics.boxHeight-this._helpWindow.height);
      this._listWindow.setHandler("ok",this.onOk.bind(this)); this._listWindow.setHandler("cancel",this.popScene.bind(this));
      this.addWindow(this._listWindow);
      this._helpWindow.setText("出力したいコモンイベントを選択してください（Enterで出力 / Escで終了）"); }
    onOk(){ const id=this._listWindow.currentId(); if(id>0){ exportCommonEvents({mode:"one",id}); this._listWindow.activate(); } else { this.popScene(); } }
  }
  class Window_CommonEventList extends Window_Selectable{
    initialize(x,y,w,h){ super.initialize(new Rectangle(x,y,w,h)); this.refresh(); this.select(0); this.activate(); }
    maxItems(){ return this._data?this._data.length:0; }
    currentId(){ const item=this._data?.[this.index()]; return item?item.id:0; }
    refresh(){ this._data=[];
      for(let i=1;i<$dataCommonEvents.length;i++){ const ce=$dataCommonEvents[i]; if(ce) this._data.push({id:i,name:ce.name||"(no name)"}); }
      this.createContents(); for(let i=0;i<this._data.length;i++) this.drawItem(i);
    }
    drawItem(index){ const rect=this.itemRect(index), item=this._data[index];
      const text=`${String(item.id).padStart(4,"0")}  ${item.name}`;
      this.drawText(text,rect.x+6,rect.y,rect.width-12);
    }
    itemHeight(){ return Math.max(super.itemHeight(),36); }
  }
})();

